/*
 * ftpToWebArchive.jsx
 * Javascript for InDesign CC, CS6, CS5.5, CS5
 * Version date: 20200225
 *
 * PURPOSE:
 * ========
 * Control BatchXSLT to FTP exported data into the web archives.
 * Data contained in the source folder ~/Export/out and its sub-structure is copied 
 * to the target domain. The sub-structure from the source folder is recreated
 * at the target domain.
 * This means:
 * If the source folder stucture is:
 *                   ~/Export/out/MyCompanyName/myNewspaper/ObjectShortcut/2008/20080607
 * the same structure is created at the domain:
 *      www.domain.com/EPAPERDATA/MyCompanyName/myNewspaper/ObjectShortcut/2008/20080607
 * 
 * the target path is build from the 'FTP Domain Name' field and the 'Data Root Path' field.
 * The subpath automatically is build to mirror the subpath below 'Source Data Path'.
 *
 * The settings from the main menu are stored in the file 'zz_Settings/settingsFTP.set'.
 * These settings are reloaded every time this program is started.
 *
 *
 * IMPORTANT NOTES
 * ===============
 * none
 *
 * DISCLAIMER:
 * ===============
 * Absolutely no warranty. Use it as is or modify it to match your needs
 *
 * Author: Andreas Imhof, www.aiedv.ch
 *
 * History:
 *    20200225 - Ai: Changed to control the freeware version
 *		20200123 - Ai: Runs with InDesign 2020 on Catalina too
 *		20171025 - Ai: Fixed a problem with font conversion tables
 * 		20170223 - Ai: General package new version 23.06.44
 * 		20150908 - Ai: Fixed: Clean image meta data from illegal XML characters
 * 		20150808 - Ai: new FTP client with FTP or FTPS
 * 		20140802 - Ai: Support CC 2014 v 10.x.x.x
 * 		20140708 - Ai: added expandable hooks at certain points
 *						afterInit, beforeFTPpush, afterFTPpush,
 *						beforeInitMainDialog, beforeShowMainDialog, afterShowMainDialog
 *
 */

// ****** Configurable SETTINGS ******
var copyright = "\u00a9www.AiEDV.ch";
var version = "40.0 46 - CC-CS5";   // Transformer and Scripts version
var packageExtension = "";        // set to an extension like '_PE' to a dedicated package installation or empty for the standard package installation
                                    // this expands the main application installation path to
                                    // BXSLT4InDesignBasePath = "~/applicationProgramName" + packageExtension + "/tranformerName/"
var tranformerName = "BatchXSLT";
var applicationHumanName = tranformerName + " for InDesign";
var applicationProgramName = tranformerName + "4InDesign";
var BXSLT4InDesignAppl = tranformerName + ".app";
var folderSeparator = "/";	// default to OSX
var os = "";
var programFolder = "";
getOS();

var	appVersion = app.version;
var majorINDVersion = 7;	// assume CS5

var settingsFileName = "settingsFTP.set";  // name of the settings file to load. default = "settingsFTP.set"
var settingsFilePath = "~/" + applicationProgramName + "Comm" + packageExtension + "/Scripts/zz_Settings";	// we must store settings somewhere into user home to have write access
var settingsFilePathAlt = "zz_Settings";							// if we have a problem try InDesign Application's script folder.

                          // The default path for XML data exported from InDesign
                          // (a path starting with tilde '~' points to the users home directory)
                          // -- set it to any existing path which should act as default export folder
                          //    and no folder chooser dialog will be shown
                          // -- set to "" to show a dialog to choose the export folder
var mainExportFolder = "";

var sourcePathName = "";
var outputPathName = "";	// this will be the base path like: ftp://site.com/_Archive/DATA/
var xslPathName = "*NONE*";

var ftpUsername = "";
var ftpPassword = "";
var ftpPort = "21";
var ftpEncoding = "UTF-8";
var includeXSLCSSfolderCopy = 1;	// set to 1 to copy XSLCSS folder too, to 0 to exclude this folder
									// tranformer can not overwrite existing files! But will copy the missing ones
var domain = "ftp://";		// the FTP domain like: ftp://site.com
var root_datapath = "";		// the data path within domain like: _Archive/DATA/


var saveSettings = true;                // true to save settings when starting export
var silent = false;                     // true to show NO completion message(s), false otherwise

                                        // The path to the folder where InDesign and tranformer will communicate 
var BXSLT4InDesignCommDir = "~/" + applicationProgramName + "Comm" + packageExtension + "/" + tranformerName + "/";

var BXSLT4InDesignBasePathDefault = programFolder + "/" + applicationProgramName + packageExtension + "/" + tranformerName + "/";	// without the root/ProgramFilesFolder
var BXSLT4InDesignBasePathDefault2 = "~/" + applicationProgramName + packageExtension + "/" + tranformerName + "/";
loadCommPrefsFromFile();
var BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault;
var BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;
                                        // The path to the folder where InDesign and tranformer will communicate 
var	startTransformer = true;

// ****** END OF Configurable SETTINGS ******
var windowTitle = applicationHumanName+ " - FTP Transfer  v" + version + "    '" +  settingsFileName + "'    " + copyright;

// ERROR codes
var last_errorCode = 0,
	error = {
		text : "",
		lastcode : 0,
		NOERR : 0,
		WRONG_INDD_VERSION : -1,
		EXPORTPATH_NOT_FOUND : -2,
		EXPORT_CANCELLED : -12
	};

var silentMode = 0;                     // Flags to combine for desired Messages
                                        // -1 = absolutely no message and dialog
                                        // 0 = show main export dialog and processing and completion messages
                                        // 1 = no main dialog


var mainDialogErrorMessage = "";
var mainDialogErrorCode = 0;

var BXSLTstarted = false;

var scriptsPath = getScriptsPath(),
	scriptName = getScriptName();

var useHooks = true,			// do check for hooks to call
	hooksBasePath = scriptsPath + folderSeparator + "zz_Plugins" + folderSeparator + "hooksFTP" + folderSeparator,			// the base path to all hooks folders
	hooksPath = "",				// the path to certain hooks folders
	hooksFolder = null,			// the hooks folder object
	hooksScripts = null,		// the hooks folder scripts files list
	hooks = {
		/* object of this type:
		"settingsFileName": {		// hook for certain settings
								"callerscriptname":scriptName,	// the caller script name
								"hook":"hook name",				// the hook name
								"args":null,	// arguments passed to scripts called by doScript (the doScripts arguments param is not working)
												// can be of any type, the callee must know how to handle it. usually an object.
								"retval":0		// the return value set by a script called by doScript (the doScripts does not return anything)
							}
		*/
	},

	ftpPushReRun = true,
	ftpPushRun = 0;


// determine a writable settings folder (in application or in home)
settingsFilePath = settingsFilePathWritable();
// load previously saved settings
loadSettingsFromFile();




// initialize and start export
app.scriptPreferences.userInteractionLevel = UserInteractionLevels.interactWithAll;

// allow hook to modify initialization
runHooks(scriptName,"afterInit", null);

/****************************
 * init some stuff.....let's go
 */
while (ftpPushReRun == true) {
	var goahead = false;

	ftpPushReRun = false;
	ftpPushRun++;

	startTransformer = true;

	// run hooks before FTP push
	runHooks(scriptName,"beforeFTPpush", null);

	goahead = init();
	//alert("goahead: " + goahead);

	// run hooks after FTP push
	if (goahead == 1) {

		// write override jobticket
		var jobticket_written = newJobTicketBatchXSLT(BXSLT4InDesignCommDir);
		//alert("jobticket_written: " + jobticket_written + "\nsourcePathName: " + sourcePathName);
		// run hooks before FTP push
		runHooks(scriptName,"afterFTPpush", null);	// run hooks after FTP push

		// fire tranformer to push XML into full-text database
		if (jobticket_written && startTransformer) {
			//alert("startTransformer\nsettingsFileName: " + settingsFileName);
			var called = callBXSLT_push();
		}
	}

}


exit();

/* ==================== FUNCTIONS ==================== */

/****************************
 * get operating system
 */
function getOS() {
	os = $.os;
	do {
		if (os.toLowerCase().indexOf("macintosh") >= 0) { 
			os = "mac";
			folderSeparator = "/";
			programFolder = folderSeparator + "Applications";
			BXSLT4InDesignAppl = tranformerName + ".app";
			break;
		}
		if (os.toLowerCase().indexOf("wind") >= 0) {
			os = "win";
			folderSeparator = "\\";
			programFolder = $.getenv("ProgramFiles");
			BXSLT4InDesignAppl = tranformerName + ".bat";
			break;
		}
		os = "unix";
		folderSeparator = "/";
		programFolder = folderSeparator + "Applications";
		BXSLT4InDesignAppl = tranformerName + ".sh";
	} while(false);
}


/****************************
 * hooks functions
 */
function getHookretval(settingsfilename) {
	if (!hooks[settingsFileName]) return 0;
	return 0;
}
function runHooks(callerscriptname, whichhook, args) {
	var hi;
	if (callerscriptname == "") {
		alert("Hooks callerscriptname is not set!! Hook call aborted!");
		return -1;
	}
	if (useHooks) {
		// export is DONE: check for hook to run after export
		hooksPath = hooksBasePath + whichhook;
		hooksFolder = new Folder(hooksPath);
		if (hooksFolder.exists) {
			hooksScripts = hooksFolder.getFiles("*.jsx*");	// get .jsx and .jsxbin
			for (hi = 0; hi < hooksScripts.length; hi++) {
				//alert("caller: " + callerscriptname + "\nhook " + whichhook + "\nscript: " + hooksScripts[hi].fsName);
				if (startsWith(hooksScripts[hi].name,"__") == false) {	// skip those starting with two underscores
					hooks[settingsFileName] = {"callerscriptname":callerscriptname,"hook":whichhook,"args":args,"retval":0};
					app.doScript(hooksScripts[hi],ScriptLanguage.javascript);
				}
			}
		}
	}
	return 0;
}


/****************************
 * init some stuff.....
 */
function init() {
	var goahead = 1;	// 0 = cancelled dialog
							// 1 = do DB push
							// 2 = load new settings
							// 3 = set factory settings

	// run with CS5 7.x.x, CS6 8.x.x, CC 9.x.x, CC 2014 10.x.x
	var versParts = appVersion.split(".");
	majorINDVersion = parseInt(versParts[0],10);
	if (majorINDVersion < 7) {
		alert("This export script may not run with this version of InDesign!\nIt is designed to work with InDesign CS5 !\n\nProcessing aborted.");
		last_errorCode = error.WRONG_INDD_VERSION;
		return(error.WRONG_INDD_VERSION);
	}


	// check if tranformer application may be found
	var bxsltappl_found = findBXSLTappl();
	if (bxsltappl_found == false) return -1;

	build_paths();

	// run hooks before FTP push
	runHooks(scriptName,"beforeInitMainDialog", null);

	if (getHookretval(settingsFileName) != -1) {	// show main dialog if not -1 is returned
		// show main dialog
		if (!(silentMode & 1)) {
			while (true) {
				goahead = showMainDialog();
				switch (goahead) {
					case 0: return(error.EXPORT_CANCELLED);	// cancelled
					case 2:	// load settings file
						break;
					case 3:	// save settings
						break;
					case 4:	// set factory settings
						break;
					default: break;	// go, do export
				}
				//alert("goahead: " + goahead);
				if ((goahead == 2) || (goahead == 3) || (goahead == 4)) continue;	// revert or load default settings
				var fieldsCheck = checkDialogFields();
				if (fieldsCheck != 0) {
					continue;
				}
				break;
			}
		}
	}

	// check if Source data folder structure may be found
	if (sourcePathName != "") {
		var f = new File(sourcePathName);
		if (f.exists == false) {
			alert("The source data folder could not be found at:\n" + sourcePathName + "\n\nPlease, verify your data.");
			return -2;
		}
	}

	return goahead;
}


/****************************
 * start transformer
 */
function callBXSLT_push() {

	// start Transformer
	BXSLTstarted = false;
	if (BXSLT4InDesignPath != "") {
		var app2start = new File (BXSLT4InDesignPath);
		BXSLTstarted = app2start.execute();
		if (BXSLTstarted == false) {
			alert(applicationHumanName + " could not be started!\nStart it manually!!");
			return(false);
		}
		else {
		}
	}

	return(true);
}


function loadCommPrefsFromFile() {
	// load the settings to communicate with tranformer from disk
	var commPrefsFile = new File(BXSLT4InDesignCommDir + "/comm/comm.prefs");
	if (commPrefsFile.exists == false) return;
	var err = commPrefsFile.open("r");

	var line = "";
	while (!commPrefsFile.eof) {
		line = commPrefsFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			//alert("line: " + line + "\nkey: " + args_arr[0] + "\narg: " + args_arr[1]);
			switch(args_arr[0]) {
				case "BatchXSLTMajorVersion": BatchXSLTMajorVersion = args_arr[1]; break;
				case "BatchXSLTMinorVersion": BatchXSLTMinorVersion = args_arr[1]; break;
				case "BatchXSLTRequiredXslVersionName": BatchXSLTRequiredXslVersionName = args_arr[1]; break;
				case "BatchXSLTAppDir": BXSLT4InDesignBasePathDefault = args_arr[1]; break;
				case "BatchXSLTCommDir": BXSLT4InDesignCommDir = args_arr[1]; break;
			}
		}
	}
	err = commPrefsFile.close();
}


/****************************
 * check if tranformer application may be found
 */
function findBXSLTappl() {
	var found = false,
		runcnt = 0;
	do {
		runcnt++;
		switch (runcnt) {
			case 1:	// try the path currently set by loadCommPrefsFromFile
				// BXSLT4InDesignBasePath already IS set
				break;
			case 2:	// try default path in user.home - may be a downloaded version for testing only
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault;
				break;
			case 3:	// try default path in /Applications (OSX) or %Programfiles% (WIN)
				BXSLT4InDesignBasePath = BXSLT4InDesignBasePathDefault2;
				break;
			case 4:	// try default path in %Programfiles% (WIN) without ' (x86)'
				if (os == 'win') {
					var bp = BXSLT4InDesignBasePathDefault2
					if (bp.indexOf(' (x86)') >= 0) bp = bp.replace(' (x86)','');
					BXSLT4InDesignBasePath = bp;
				} else continue;
				break;
			case 5:	// try to find any folder in /Applications containing 'applicationProgramName*'
				var fldr = new Folder(programFolder);
				var applFldrs = fldr.getFiles(applicationProgramName + packageExtension + "*");
				if (applFldrs.length < 1) break;	// let user choose the application
				for (var i = 0; i < applFldrs.length; i++) {
					var afldr = applFldrs[i];
					var applfldr = afldr + folderSeparator + tranformerName + folderSeparator + tranformerName + ".app";
					//alert(applfldr);
					fldr = new File(applfldr);
					if (fldr.exists == true) {
						BXSLT4InDesignBasePath = afldr + folderSeparator + tranformerName + folderSeparator;
						//alert("found appl: " + applfldr);
						break;
					}
				}
				break;	// let user choose the application
			default:	// this will open the file chooser
		}
		// set transformer application to call
		BXSLT4InDesignPath = BXSLT4InDesignBasePath + BXSLT4InDesignAppl;
		//alert("runcnt:"+runcnt +"\n"+ BXSLT4InDesignPath);
		var f = new File(BXSLT4InDesignPath);
		if (f.exists == false) {
			if (runcnt < 5) continue;
			alert("The '" + applicationHumanName + "' application can not be found at\n" + BXSLT4InDesignPath + "\n\nPlease, copy the application to your HOME folder (as mentioned in the manual) or choose the application in the following dialog and try again!");
			var bxslt_appl_fldr = Folder.selectDialog("Please select the folder containing the tranformer application");	// "Please select the folder containing the tranformer application"
			if (bxslt_appl_fldr == null) {
				alert("FTP Push aborted.\n\nThe application may not continue because the Transformer application may not be found.");
				return(false);
			}
			if (bxslt_appl_fldr != null) BXSLT4InDesignBasePath = bxslt_appl_fldr + folderSeparator;
			continue;
		}
		found = true;
		break;
	} while(true);
	return(found);
}




/****************************
 * create a jobticket for tranformer
 */
function newJobTicketBatchXSLT(path) {
	var jtname = new Date().getTime() + "override.jt",
		jtFile = null,
		commpath = path,
		f, fldr,
		err;

	if (path == null || path == "") return;
	f = new File(path);
	if (f.exists == false) {
		fldr = new Folder(path);
		fldr.create();
		f = new File(path);
		if (f.exists == false) {
			alert("## An error ocurred while creating the communication folder\n" + BXSLT4InDesignCommDir + "\n\nTransfer failed");
			return;
		}
	}

	//-----------------------------------------
	// write a JobTicket to FTP the files to the server archives
	my_sourcePathName = sourcePathName.replace(/\\/gi,"/");

	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	jtFile = new File(commpath + jtname);
	jtFile.encoding = "UTF-8";
	err = jtFile.open("w");

	err = jtFile.write("init_previous_jobticket = 1\n");
	err = jtFile.write("jobticketVersion = 10.0\n");
	err = jtFile.write("mode = 0\n");
	err = jtFile.write("logfileWrite = 1\n");
	err = jtFile.write("\n");
	err = jtFile.write("jt_triggerfile = \n");
	err = jtFile.write("sourcePathName = " + my_sourcePathName + "\n");
	err = jtFile.write("xslPathName = *NONE*\n");
	err = jtFile.write("outputPathName = " + outputPathName + "\n");
	err = jtFile.write("newoutputFileNameExt = \n");
	err = jtFile.write("sourceFileAction = \n");
	err = jtFile.write("deleteSourceDirs = 0\n");
	err = jtFile.write("nextJobTicketPath =\n");
	err = jtFile.write("nextJobTicketFileName = autostart.jt\n");

	err = jtFile.write("includeXSLCSSfolderCopy = " + includeXSLCSSfolderCopy + "\n");
	if (includeXSLCSSfolderCopy == 0) {
		err = jtFile.write("excludeCleanupRunFileNameExts = XSLCSS\n");
		err = jtFile.write("excludeSourceFileNameExts = XSLCSS\n");
	}
	else {
		err = jtFile.write("excludeCleanupRunFileNameExts =\n");
		err = jtFile.write("excludeSourceFileNameExts =\n");
	}

	err = jtFile.write("\n");
	err = jtFile.write("ftpUsername = " + ftpUsername + "\n");
	err = jtFile.write("ftpPassword = " + ftpPassword + "\n");
	err = jtFile.write("ftpPort = " + ftpPort + "\n");
	err = jtFile.write("ftpEncoding = " + ftpEncoding + "\n");
	err = jtFile.write("\n");

	err = jtFile.close();

	writeOverrideQueue(commpath,jtname);

	return(err);
}


function writeOverrideQueue(path,jtname,jtname1) {
	if ((path == null) || (path == "")) return(-1);
	var commpath = path;
	if (endsWith(commpath,folderSeparator) == false) commpath += folderSeparator;
	var queuename = "override.que";
	var queuepath = commpath + queuename;
	var xqueuename = "x" + queuename;
	var xqueuepath = commpath + xqueuename;
	jtQfile = new File(queuepath);
	var err = false;
	if (jtQfile.exists == true) {	// try to rename queue file if exists
		for (var i = 0; i < 1000; i++) {	// try to rename file
			err = jtQfile.rename(xqueuename);
			if (jtQfile.error != "") ;	//alert("rename fserror: " +jtQfile.error);
			if (err == true) break;
			//alert("locked");
			$.sleep(10);
		}
		if (err == false) {
			//alert("error rename to: " + xqueuename);
			return(-1);	// uuups
		}
	}

	jtQfile = new File(xqueuepath);
	jtQfile.encoding = "UTF-8";
	for (var i = 0; i < 100; i++) {	// try to open until file is free. don't wait too long
		err = jtQfile.open("e");
		if (err == true) break;
		$.sleep(100);
	}
	if (err == false) return(-1);
	if (jtQfile.error != "") alert("fserror: " +jtQfile.error);

	err = jtQfile.seek(0,2);	// seek to eof to append
	//if (err != true) alert("error seek");
	if ((jtname != null) && (jtname != "")) err = jtQfile.writeln(jtname);
	//if (err != true) alert("error jt write");
	if ((jtname1 != null) && (jtname1 != "")) err = jtQfile.writeln(jtname1);
	//if (err != true) alert("error jt1 write");
//	alert("wait to close file");
	err = jtQfile.close();
	//if (err != true) alert("error close");
	err = jtQfile.rename(queuename);
	//if (err != true) alert("error rename back to: " + queuename);
	if (err == false) return(-1);
	return(0);
}




/****************************
 * SHOW THE STARTING MAIN DIALOG
 */
function showMainDialog() {
	var outfldr = new Folder(sourcePathName),
		trans = new Folder(BXSLT4InDesignBasePath),
		dlgw,

		windowFont,
		font,

		leftGroupWidth,
		leftGroupSiz,
		spacerSize1,
		labelSize,
		editFieldSize,
		editFieldSize2,
		dotdotWidth = 60;

	windowTitle = applicationHumanName + " - FTP Transfer  v" + version + "    '" +  settingsFileName + "'    " + copyright;
	dlgw = new Window('dialog', windowTitle, undefined, {resizeable:false, closeButton: true, maximizeButton:false, minimizeButton:false});
	dlgw.orientation = 'row';
	dlgw.margins = [7,7,7,7];
	dlgw.alignChildren = ['left','top'];
	dlgw.alignment = ['left','top'];

	windowFont = dlgw.graphics.font;
	font = ScriptUI.newFont ( windowFont.name, windowFont.name.style, 11 );
	leftGroupWidth = 600;
	leftGroupSize = [leftGroupWidth,300];
	spacerSize1 = [10,15];
	labelSize = [130,2.0*font.size];
	labelRowSize = [300,3.0*font.size];
	editFieldSize = [150,2.0*font.size];
	editFieldSize2 = [450,2.0*font.size];

	with(dlgw){
		//------------------------------
		// items on left side
		leftGroup = add('group');
		leftGroup.orientation = 'column';
		leftGroup.alignment = ['left','top'];
		leftGroup.alignChildren = ['left','top'];
		leftGroup.spacing = 5;

		with(leftGroup) {
			if (mainDialogErrorMessage != "") {	// show the error message
				var errmessage = "Error #" + mainDialogErrorCode + " in Dialog fields: " + mainDialogErrorMessage;
				errormess = add('statictext', undefined, errmessage);
			}

			sourcePathGroup = add('group', undefined);
			sourcePathGroup.orientation = 'row';
			sourcePathGroup.alignChildren = ['left','top'];
			sourcePathGroup.alignment = ['left','top'];
				sourcePathGroup.label = sourcePathGroup.add('statictext', undefined, "Source Data Path:");
					sourcePathGroup.label.graphics.font = font;
					sourcePathGroup.label.size = labelSize;
					sourcePathGroup.label.justify = "right";
				sourcePathGroup.sourcePathNameField = sourcePathGroup.add('edittext', undefined, sourcePathName);
					sourcePathGroup.sourcePathNameField.size = editFieldSize2;
					sourcePathGroup.sourcePathNameField.graphics.font = font;
					sourcePathGroup.sourcePathNameField.helpTip = "The path to the source data to transfer";

			// ------------------------- a spacer
			add('statictext', undefined, "").size = spacerSize1;

			// items on left side in left groug
			ll = add('group').add('panel', undefined, '  Transfer To  ');
			ll.orientation = 'column';
			ll.alignment = ['left','top'];
			ll.alignChildren = ['left','top'];
			ll.spacing = 5;

			with(ll) {
				ftpURLGroup = add('group', undefined);
				ftpURLGroup.alignment = ['left','top'];
				ftpURLGroup.orientation = 'row';
				ftpURLGroup.alignChildren = ['right','top'];
					ftpURLGroup.urlLabel = ftpURLGroup.add('statictext', undefined, "FTP Server URL:");
						ftpURLGroup.urlLabel.graphics.font = font;
						ftpURLGroup.urlLabel.size = labelSize;
						ftpURLGroup.urlLabel.justify = "right";
					ftpURLGroup.ftp_URLField = ftpURLGroup.add('edittext', undefined, domain);
						ftpURLGroup.ftp_URLField.size = editFieldSize2;
						ftpURLGroup.ftp_URLField.graphics.font = font;
						ftpURLGroup.ftp_URLField.helpTip = "The URL to the FTP server like'ftp://mydomain.com/";

				ftpRootPathGroup = add('group', undefined);
				ftpRootPathGroup.alignment = ['left','top'];
				ftpRootPathGroup.orientation = 'row';
				ftpRootPathGroup.alignChildren = ['right','top'];
					ftpRootPathGroup.root_datapathLabel = ftpRootPathGroup.add('statictext', undefined, "Data Root Path:");
						ftpRootPathGroup.root_datapathLabel.graphics.font = font;
						ftpRootPathGroup.root_datapathLabel.size = labelSize;
						ftpRootPathGroup.root_datapathLabel.justify = "right";
					ftpRootPathGroup.root_datapathField = ftpRootPathGroup.add('edittext', undefined, root_datapath);
						ftpRootPathGroup.root_datapathField.size = editFieldSize2;
						ftpRootPathGroup.root_datapathField.graphics.font = font;
						ftpRootPathGroup.root_datapathField.helpTip = "The path to fill into the 'root_datapath' field in the database table";

				ftpUserGroup = add('group', undefined);
				ftpUserGroup.alignment = ['left','top'];
				ftpUserGroup.orientation = 'row';
				ftpUserGroup.alignChildren = ['right','top'];
					ftpUserGroup.ftp_UsernameLabel = ftpUserGroup.add('statictext', undefined, "User Name:");
						ftpUserGroup.ftp_UsernameLabel.graphics.font = font;
						ftpUserGroup.ftp_UsernameLabel.size = labelSize;
						ftpUserGroup.ftp_UsernameLabel.justify = "right";
					ftpUserGroup.ftp_UsernameField = ftpUserGroup.add('edittext', undefined, ftpUsername);
						ftpUserGroup.ftp_UsernameField.size = editFieldSize;
						ftpUserGroup.ftp_UsernameField.graphics.font = font;
						ftpUserGroup.ftp_UsernameField.helpTip = "The FTP server login User Name";

					ftpUserGroup.ftp_PasswordLabel = ftpUserGroup.add('statictext', undefined, "Password:");
						ftpUserGroup.ftp_PasswordLabel.graphics.font = font;
						ftpUserGroup.ftp_PasswordLabel.size = labelSize;
						ftpUserGroup.ftp_PasswordLabel.justify = "right";
					ftpUserGroup.ftp_PasswordField = ftpUserGroup.add('edittext', undefined, ftpPassword);
						ftpUserGroup.ftp_PasswordField.size = editFieldSize;
						ftpUserGroup.ftp_PasswordField.graphics.font = font;
						ftpUserGroup.ftp_PasswordField.helpTip = "The FTP server login User Password";
			}

			// ------------------------- a spacer
			add('statictext', undefined, "").size = spacerSize1;

			ftpFullPathGroup = add('group', undefined);
			ftpFullPathGroup.alignment = ['left','top'];
			ftpFullPathGroup.orientation = 'row';
			ftpFullPathGroup.alignChildren = ['left','top'];
				ftpFullPathGroup.ftp_fullPathLabel = ftpFullPathGroup.add('statictext', undefined, "FTP Full Path:");
					ftpFullPathGroup.ftp_fullPathLabel.graphics.font = font;
					ftpFullPathGroup.ftp_fullPathLabel.size = labelSize;
				ftpFullPathGroup.ftp_fullPathField = ftpFullPathGroup.add('statictext', undefined, dotText(outputPathName, dotdotWidth));
					ftpFullPathGroup.ftp_fullPathField.graphics.font = font;
					ftpFullPathGroup.ftp_fullPathField.justify = "left";
					ftpFullPathGroup.ftp_fullPathField.size = editFieldSize2;

			ftptransformerPathGroup = add('group', undefined);
			ftptransformerPathGroup.alignment = ['left','top'];
			ftptransformerPathGroup.orientation = 'row';
			ftptransformerPathGroup.alignChildren = ['left','top'];
				ftptransformerPathGroup.ftp_transformerPathLabel = ftptransformerPathGroup.add('statictext', undefined, "Transformer Path:" );
					ftptransformerPathGroup.ftp_transformerPathLabel.graphics.font = font;
					ftptransformerPathGroup.ftp_transformerPathLabel.size = labelSize;
				ftptransformerPathGroup.ftp_transformerPathField = ftptransformerPathGroup.add('statictext', undefined, trans.fsName );
					ftptransformerPathGroup.ftp_transformerPathField.graphics.font = font;

			includeXSLCSSfolderCopyCheck = add("checkbox", undefined, "Transfer XSLCSS folder");
				includeXSLCSSfolderCopyCheck.value = includeXSLCSSfolderCopy;
				includeXSLCSSfolderCopyCheck.graphics.font = font;
				includeXSLCSSfolderCopyCheck.helpTip = "Check to also transfer the folder 'XSLCSS'";

			saveSettingsCheck = add("checkbox", undefined, "Save these settings");
				saveSettingsCheck.value = saveSettings;
				saveSettingsCheck.graphics.font = font;
				saveSettingsCheck.helpTip = "Save these settings to a file on disk";

		}




		//------------------------------
		// The buttons on right side
		rightGroup = dlgw.add('group');
		rightGroup.orientation = 'column';
		rightGroup.alignChildren = ['fill','top'];
		rightGroup.alignment = ['fill','top'];
			rightGroup.btnPanel = rightGroup.add('panel', undefined, '');
				rightGroup.btnPanel.goBtn = rightGroup.btnPanel.add('button', undefined, "GO", {name:'go'});
				rightGroup.btnPanel.goBtn.helpTip = "Start FTP transfer using the current settings";
				rightGroup.btnPanel.cancelBtn = rightGroup.btnPanel.add('button', undefined, "Cancel", {name:'cancel'});
				rightGroup.btnPanel.cancelBtn.helpTip = "Cancel FTP transfer";

			rightGroup.sub2Group = rightGroup.add('group');
			rightGroup.sub2Group.orientation = 'column';
			rightGroup.sub2Group.alignChildren = ['fill','top'];
			rightGroup.sub2Group.alignment = ['fill','top'];
			rightGroup.sub2Group.margins = [0,38,0,0];
				rightGroup.sub2Group.loadSettingsFromFileBtn = rightGroup.sub2Group.add('button', undefined, "Load Settings");
				rightGroup.sub2Group.loadSettingsFromFileBtn.helpTip = "Load new settings from settings file";	// Load new settings from settings file
				rightGroup.sub2Group.saveSettingsToFileBtn = rightGroup.sub2Group.add('button', undefined, "Save Settings");
				rightGroup.sub2Group.saveSettingsToFileBtn.helpTip = "Save settings to file";	// Save settings to file

		/*
			The dialog item handlers
		*/
		// Go and Cancel buttons
		rightGroup.btnPanel.goBtn.onClick = function() { dlgw.close(1); } 
		rightGroup.btnPanel.cancelBtn.onClick = function() { dlgw.close(0); } 
		// Settings buttons
		rightGroup.sub2Group.loadSettingsFromFileBtn.onClick = function() {
			var f = new Folder(settingsFilePath);
			var selected = f.openDlg("Choose Settings File");
			if (selected != null) {
				var fnew = new Folder(selected);
				settingsFilePath = fnew.path;
				settingsFileName = fnew.name;
				//alert("selected: " + selected + "\nfullpath: " + fnew.fsName + "\npath: " + fnew.path + "\nname: " + fnew.name);
				var loaded = loadSettingsFromFile();
				dlgw.close(2);
			}
		} 
		rightGroup.sub2Group.saveSettingsToFileBtn.onClick = function() {
			var savepath = settingsFilePath;
			if (endsWith(savepath,folderSeparator) == false) savepath += folderSeparator;
			savepath += settingsFileName;
			var f = new File(savepath);
			var selected = f.saveDlg("Name the Settings File");
			if (selected != null) {
				settingsFilePath = selected.path;
				settingsFileName = selected.name;
				settingsloaded = 0;
				if (endsWith(settingsFileName.toLowerCase(),".set") == false) settingsFileName += ".set";
				//alert("savepath: " + savepath + "\nselected: " + selected + "\nsettingsFilePath: " + settingsFilePath + "\nsettingsFileName: " + settingsFileName);
				dlgw.close(3);
			}
		} 


		function showFullFTPpath () {
			var url = ftpURLGroup.ftp_URLField.text,
				rootpath = ftpRootPathGroup.root_datapathField.text;
			if ( (endsWith(url,"/") == false) && (startsWith(rootpath,"/") == false)) url += "/";
			url += rootpath;
			if (endsWith(url,"/") == false) url += "/";

			ftpFullPathGroup.ftp_fullPathField.text = dotText(url, dotdotWidth);
		}

		ftpURLGroup.ftp_URLField.onChanging = function() { showFullFTPpath(); };
		ftpRootPathGroup.root_datapathField.onChanging = function() { showFullFTPpath(); };
	}	


	// before show main dialog: run hook
	runHooks(scriptName,"beforeShowMainDialog", null);

	//------------------------------
	// Display the dialog box.
	var go = dlgw.show();
	if ((go == 2) || (go == 4)) return(go);	// revert or load default settings

	// Get all fields
	sourcePathName = sourcePathGroup.sourcePathNameField.text;
	domain = ftpURLGroup.ftp_URLField.text;
	root_datapath = ftpRootPathGroup.root_datapathField.text;
	ftpUsername = ftpUserGroup.ftp_UsernameField.text;
	ftpPassword = ftpUserGroup.ftp_PasswordField.text;

	outputPathName = domain;
	if ( (endsWith(outputPathName,"/") == false) && (startsWith(root_datapath,"/") == false)) outputPathName += "/";
	outputPathName += root_datapath;
	if (endsWith(outputPathName,"/") == false) outputPathName += "/";

	if (includeXSLCSSfolderCopyCheck.value == true) {
		includeXSLCSSfolderCopy = 1;
	}
	else includeXSLCSSfolderCopy = 0;


	// should current settings be saved?
	if ((go == 3) || (saveSettingsCheck.value == true)) {
		saveSettings = true;
		saveSettingsToFile();
	}
	else saveSettings = false;

	try { dlgw.close();} catch(e) {}

	// after show main dialog: run hook
	runHooks(scriptName,"afterShowMainDialog", {"go":go});

	return(go);
}

function dotText(text, length) {
	if (text == null || text == "") return(text);
	if (length <= 5) return("");
	if (text.length <= length) return(text);
	var str1 = text.substring(0,parseInt(length/2)-3);
	var str2 = text.substring(text.length-length/2);
	return(str1 + "..." + str2);
}

function checkDialogFields() {
	mainDialogErrorMessage = "";
	mainDialogErrorCode = 0;

	build_paths();	// build target ftp path

	if ( sourcePathName == "" ) {
		mainDialogErrorMessage = "'Source Data Path' may not be empty";
		mainDialogErrorCode = 1;
		return(mainDialogErrorCode);
	}
	if ( domain == "" ) {
		mainDialogErrorMessage = "'FTP Domain Name' may not be empty";
		mainDialogErrorCode = 2;
		return(mainDialogErrorCode);
	}
	if ( ftpUsername == "" ) {
		mainDialogErrorMessage = "'FTP User Name' may not be empty";
		mainDialogErrorCode = 3;
		return(mainDialogErrorCode);
	}
	if ( root_datapath == "" ) {
		mainDialogErrorMessage = "'Data Root Path' may not be empty";
		mainDialogErrorCode = 4;
		return(mainDialogErrorCode);
	}
	if ( ftpPassword == "" ) {
		mainDialogErrorMessage = "'FTP Password' may not be empty";
		mainDialogErrorCode = 5;
		return(mainDialogErrorCode);
	}
	if ( outputPathName == "" ) {
		mainDialogErrorMessage = "'FTP URL' may not be empty";
		mainDialogErrorCode = 6;
		return(mainDialogErrorCode);
	}
	if ( (outputPathName.indexOf("ftp://") < 0)
		&& (outputPathName.indexOf("ftps://") < 0)
		) {
		mainDialogErrorMessage = "'FTP URL' MUST start with the protocol identifier 'ftp://' or 'ftps://'";
		mainDialogErrorCode = 7;
		return(mainDialogErrorCode);
	}


	return(mainDialogErrorCode);
}






function settingsFilePathWritable() {	// is settings path writable? otherwise return alternate settings path
	var fldr = new Folder(settingsFilePath),	// make sure the settings path exists
		created = fldr.create();
	//alert("settingsFilePathWritable settingsFilePath: " + settingsFilePath + "\ncreated: " + created);
	if (created == true) {	// we can write
		return(settingsFilePath);
	}
	else {
		// check if we can write into InDesign's scripting folder
		var checkFolder = new File(getScriptsPath() + "/" + settingsFilePathAlt);
		var checkFile = new File(getScriptsPath() + "/" + settingsFilePathAlt + "/writecheck");
		var created = checkFile.open("w");
		if (created == true) {	// we can write
			var settingsPath = checkFile.fsName;
			checkFile.close();
			//alert("writable: " + settingsPath);
			checkFile.remove();
			return(checkFolder.fsName);
		}
	}
	return(settingsFilePath);	// uuuups
}

function saveSettingsToFile() {
	// write the current settings to disk
	var settingsFile = new File(settingsFilePath + "/" + settingsFileName);
	var err = settingsFile.open("w");

	err = settingsFile.write("mainExportFolder=" + getURIName(mainExportFolder) + "\n");
	err = settingsFile.write("sourcePathName=" + getURIName(sourcePathName) + "\n");
	err = settingsFile.write("includeXSLCSSfolderCopy=" + includeXSLCSSfolderCopy + "\n");
	err = settingsFile.write("ftpUsername=" + ftpUsername + "\n");
	err = settingsFile.write("ftpPassword=" + ftpPassword + "\n");
	err = settingsFile.write("ftpPort=" + ftpPort + "\n");
	err = settingsFile.write("ftpEncoding=" + ftpEncoding + "\n");

	err = settingsFile.write("domain=" + domain + "\n");
	err = settingsFile.write("root_datapath=" + root_datapath + "\n");

	err = settingsFile.write("saveSettings=" + saveSettings + "\n");

	err = settingsFile.write("\n");	// ending empty line!!
	err = settingsFile.close();

}

function getURIName(path) {
	if ((path == null) || (path == "")) return("");
	try {
		var f = new File(path);
		return(f.fullName);
	} catch(e) {}
	return(path);
}


function loadSettingsFromFile() {
	// load settings from file
	//alert("loadSettingsFromFile: " + settingsFilePath + "/" + settingsFileName);
	var settingsFile = new File(settingsFilePath + "/" + settingsFileName);
	do {
		if (settingsFile.exists == false) break;
	var err = settingsFile.open("r");
	
		var line = settingsFile.readln();
		while (!settingsFile.eof) {
			if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
				var key = line.substr(0,line.indexOf("="));
				var arg = line.substr(line.indexOf("=")+1);
				var args_arr = new Array(key,arg);
				//alert("line: " + line + "\nkey: " + args_arr[0] + "\narg: " + args_arr[1]);
				switch(args_arr[0]) {
					case "mainExportFolder": mainExportFolder = args_arr[1]; break;
					case "sourcePathName": sourcePathName = args_arr[1]; break;
					case "includeXSLCSSfolderCopy": includeXSLCSSfolderCopy = parseInt(args_arr[1]); break;
					case "ftpUsername": ftpUsername = args_arr[1]; break;
					case "ftpPassword": ftpPassword = args_arr[1]; break;
					case "ftpPort": ftpPort = args_arr[1]; break;
					case "ftpEncoding": ftpEncoding = args_arr[1]; break;
					case "domain": domain = args_arr[1]; break;
					case "root_datapath": root_datapath = args_arr[1]; break;
	
					case "saveSettings": if (args_arr[1] == "true") saveSettings = true; else saveSettings = false; break;
				}
			}
			line = settingsFile.readln();
		}
		err = settingsFile.close();
	} while (false);

	if ( mainExportFolder == "") {	// get from main settings.set
		loadMainSettingsFromFile();
		if ( mainExportFolder == "") mainExportFolder = "~/Export";
	}

	if (sourcePathName == "") sourcePathName = mainExportFolder + "/out/";
	var outfldr = new Folder(sourcePathName);
	sourcePathName = outfldr.fsName;

}


function loadMainSettingsFromFile() {
	var settingsFile = new File(settingsFilePath + "/settings.set");
	if (settingsFile.exists == false) return;
	var err = settingsFile.open("r");

	var line = settingsFile.readln();
	while (!settingsFile.eof) {
		if (line != "") {
			var args_arr = line.split("=");
			switch(args_arr[0]) {
				case "mainExportFolder": mainExportFolder = args_arr[1]; break;
			}
		}
		line = settingsFile.readln();
	}
	err = settingsFile.close();
}


function build_paths() {

	outputPathName = "";
	if (domain != "") outputPathName = domain;
	if (outputPathName != "") {
		if ( (endsWith(outputPathName,"/") == false) && (startsWith(root_datapath,"/") == false)) outputPathName += "/";
		outputPathName += root_datapath;
		if (endsWith(outputPathName,"/") == false) outputPathName += "/";
	}
}


function getScriptsPath() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.path);
}
function getScriptName() {
	var fullpath = app.activeScript,
		scriptFile = new File(fullpath);
	return(scriptFile.name);
}


var messageWindow = null;
function messageClose() {
	if (!is_CS3_or_later) return;
	if (messageWindow != null) messageWindow.close();
}
function message(mess) {
	if (!is_CS3_or_later) return;
	if (messageWindow == null) messageCreate(mess);
	else messageWindow.messtext.text = mess;
	messageWindow.show();
}
function messageCreate(mess) {
	messageWindow = new Window('window', 'Progress', undefined, {maximizeButton:false, minimizeButton:false});
	messageWindow.messageGroup = messageWindow.add('group');
	messageWindow.messageGroup.orientation = 'row';
	messageWindow.messageGroup.margins = [50,30,50,30];
	messageWindow.messtext = messageWindow.messageGroup.add('statictext', undefined, mess);
}


function listProperties(the_prop_to_query) {
		var props_arr = the_prop_to_query;
		var props = the_prop_to_query.reflect.properties; 
		for (var i = 0; i < props.length; i++) { 
			alert("this property '" + props[i].name + "' is '" + props_arr[props[i].name] + "'"); 
		}
}

function startsWith (str,subs) { return (str.match("^"+subs)==subs); }

function  endsWith (str,subs) { return (str.match(subs+"$")==subs); }
